package com.pt5.pthouduan.ControllerTest;


import com.pt5.pthouduan.controller.TorrentController;
import com.pt5.pthouduan.service.TorrentService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

class TorrentDeleteControllerTest {

    @Mock
    private TorrentService torrentService;

    @InjectMocks
    private TorrentController torrentController;

    private MockMvc mockMvc;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
        mockMvc = MockMvcBuilders.standaloneSetup(torrentController).build();
    }

    @Test
    void deleteTorrent_WithPermission_ShouldReturnSuccess() throws Exception {
        // Arrange
        Long userid = 1L;
        Long torrentId = 101L;

        // 模拟服务层调用成功
        doNothing().when(torrentService).deleteTorrent(userid, torrentId);

        // Act & Assert
        mockMvc.perform(delete("/torrent/delete/{torrentId}", torrentId)
                        .param("userid", userid.toString()))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.success").value(true))
                .andExpect(jsonPath("$.message").value("种子删除成功"));

        // 验证服务层方法被调用
        verify(torrentService, times(1)).deleteTorrent(userid, torrentId);
    }

    @Test
    void deleteTorrent_WithoutPermission_ShouldReturnForbidden() throws Exception {
        // Arrange
        Long userid = 2L;
        Long torrentId = 101L;

        // 模拟服务层抛出SecurityException
        doThrow(new SecurityException("无权删除此种子")).when(torrentService).deleteTorrent(userid, torrentId);

        // Act & Assert
        mockMvc.perform(delete("/torrent/delete/{torrentId}", torrentId)
                        .param("userid", userid.toString()))
                .andExpect(status().isForbidden())
                .andExpect(jsonPath("$.success").value(false))
                .andExpect(jsonPath("$.message").value("无权删除此种子"));

        // 验证服务层方法被调用
        verify(torrentService, times(1)).deleteTorrent(userid, torrentId);
    }

    @Test
    void deleteTorrent_WhenServiceThrowsException_ShouldReturnInternalServerError() throws Exception {
        // Arrange
        Long userid = 1L;
        Long torrentId = 101L;

        // 模拟服务层抛出其他异常
        doThrow(new RuntimeException("数据库错误")).when(torrentService).deleteTorrent(userid, torrentId);

        // Act & Assert
        mockMvc.perform(delete("/torrent/delete/{torrentId}", torrentId)
                        .param("userid", userid.toString()))
                .andExpect(status().isInternalServerError())
                .andExpect(jsonPath("$.success").value(false))
                .andExpect(jsonPath("$.message").value("删除失败: 数据库错误"));

        // 验证服务层方法被调用
        verify(torrentService, times(1)).deleteTorrent(userid, torrentId);
    }

    @Test
    void deleteTorrent_WithoutUserIdParameter_ShouldReturnBadRequest() throws Exception {
        // Arrange
        Long torrentId = 101L;

        // Act & Assert
        mockMvc.perform(delete("/torrent/delete/{torrentId}", torrentId))
                .andExpect(status().isBadRequest());
    }
}