package com.pt5.pthouduan.controller;

import com.pt5.pthouduan.entity.Activity;
import com.pt5.pthouduan.service.ActivityService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

class ActivityControllerTest {

    @Mock
    private ActivityService activityService;

    @InjectMocks
    private ActivityController activityController;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void getVisibleActivityPreviews_ShouldReturnPreviewList() {
        Activity fullActivity = new Activity();
        fullActivity.setActivityid(1);
        fullActivity.setTitle("Test");
        fullActivity.setContent("Content");
        fullActivity.setAward(10);
        fullActivity.setPhoto("/activity/photo.jpg");
        fullActivity.setTime(LocalDateTime.now());

        when(activityService.findByIsShow(0)).thenReturn(Arrays.asList(fullActivity));

        List<Activity> result = activityController.getVisibleActivityPreviews();

        assertEquals(1, result.size());
        assertEquals("Test", result.get(0).getTitle());
        assertEquals("/activity/photo.jpg", result.get(0).getPhoto());
    }

    @Test
    void getVisibleActivities_ShouldReturnList() {
        Activity activity = new Activity();
        activity.setActivityid(1);
        activity.setTitle("Visible Activity");

        when(activityService.findByIsShow(0)).thenReturn(List.of(activity));

        List<Activity> result = activityController.getVisibleActivities();

        assertEquals(1, result.size());
        assertEquals("Visible Activity", result.get(0).getTitle());
    }

    @Test
    void createActivity_ShouldReturnTrue_WhenSaved() {
        when(activityService.save(any(Activity.class))).thenReturn(true);

        boolean result = activityController.createActivity(
                "New Title", "New Content", null, 0, 5
        );

        assertTrue(result);
        verify(activityService).save(any(Activity.class));
    }

    @Test
    void deleteActivity_ShouldReturnTrue_WhenDeleted() {
        when(activityService.removeById(1)).thenReturn(true);

        boolean result = activityController.deleteActivity(1);

        assertTrue(result);
        verify(activityService).removeById(1);
    }

    @Test
    void getAllActivities_ShouldReturnAllActivities() {
        Activity a1 = new Activity();
        a1.setTitle("A1");

        Activity a2 = new Activity();
        a2.setTitle("A2");

        when(activityService.list()).thenReturn(List.of(a1, a2));

        List<Activity> result = activityController.getAllActivities();

        assertEquals(2, result.size());
        assertEquals("A1", result.get(0).getTitle());
        assertEquals("A2", result.get(1).getTitle());
    }

    @Test
    void searchActivitiesByTitle_ShouldReturnMatchingActivities() {
        Activity match = new Activity();
        match.setTitle("Matched");

        when(activityService.searchByTitle("Match")).thenReturn(List.of(match));

        List<Activity> result = activityController.searchActivitiesByTitle("Match");

        assertEquals(1, result.size());
        assertEquals("Matched", result.get(0).getTitle());
    }

    @Test
    void updateActivity_ShouldReturnTrue_WhenSuccessful() {
        Activity existing = new Activity();
        existing.setActivityid(1);
        existing.setTitle("Old");

        when(activityService.getById(1)).thenReturn(existing);
        when(activityService.updateById(any(Activity.class))).thenReturn(true);

        boolean result = activityController.updateActivity(1, "New Title", "New Content", null, 1, 10);

        assertTrue(result);
        verify(activityService).updateById(any(Activity.class));
    }

    @Test
    void updateActivity_ShouldReturnFalse_WhenActivityNotFound() {
        when(activityService.getById(999)).thenReturn(null);

        boolean result = activityController.updateActivity(999, "Title", "Content", null, 0, 0);

        assertFalse(result);
    }
}
