package com.pt5.pthouduan.controller;

import com.pt5.pthouduan.entity.request_resources;
import com.pt5.pthouduan.service.RequestResourcesService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.stereotype.Controller;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * <p>
 *  求助帖信息前端控制器
 * </p>
 *
 * 功能：创建求助帖、修改金额、删除、按名称/用户查找、计算某名称总金额
 * 支持上传图片
 *
 * @author ym
 * @since 2025-05-10
 */
@CrossOrigin(origins = "http://localhost:5174")
@Controller
@RequestMapping("/request")
public class RequestResourcesController {

    @Autowired
    private RequestResourcesService requestResourcesService;

    @Value("${torrent.helppost-image-dir}")
    private String uploadDir;

    // 创建新的求助帖（移除 loaduser 参数）
    @PostMapping("/create")
    @ResponseBody
    public boolean createRequest(
            @RequestParam("userid") Long userid,
            @RequestParam("name") String name,
            @RequestParam("plot") String plot,
            @RequestParam("money") Integer money,
            @RequestParam(value = "photo", required = false) MultipartFile photoFile,
            @RequestParam(value = "year", required = false) Integer year,
            @RequestParam(value = "country", required = false) String country
    ) {
        request_resources request = new request_resources();
        request.setUserid(userid);
        request.setLoaduser(null); // 初始为空
        request.setName(name);
        request.setPlot(plot);
        request.setMoney(money);
        request.setYear(year);
        request.setCountry(country);
        request.setRequestTime(LocalDateTime.now());

        if (photoFile != null && !photoFile.isEmpty()) {
            try {
                File dir = new File(uploadDir);
                if (!dir.exists()) {
                    boolean created = dir.mkdirs();  // 递归创建目录
                    if (!created) {
                        throw new IOException("无法创建目录：" + uploadDir);
                    }
                }
                //String coverImageDir = "D:/torrenttest/covers/"; // 封面图片存储目录
                String filename = System.currentTimeMillis()+"_"+photoFile.getOriginalFilename();
                Path helprequestImagePath = Paths.get(uploadDir, filename);

                try {
                    Files.createDirectories(helprequestImagePath.getParent()); // 创建目录
                    Files.copy(photoFile.getInputStream(), helprequestImagePath, StandardCopyOption.REPLACE_EXISTING);
                } catch (IOException e) {
                    throw new RuntimeException("无法存储需求贴图片", e);
                }

                // 设置访问URL
                request.setPhoto('/' + uploadDir + filename);
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }

        return requestResourcesService.createRequest(request);
    }

    // 修改求助帖的金额
    @PutMapping("/updateMoney/{requestid}")
    @ResponseBody
    public boolean updateMoney(@PathVariable Integer requestid, @RequestParam Integer money) {
        return requestResourcesService.updateMoney(requestid, money);
    }

    // 删除求助帖
    @DeleteMapping("/delete/{requestid}")
    @ResponseBody
    public boolean deleteRequest(@PathVariable Integer requestid) {
        return requestResourcesService.deleteRequest(requestid);
    }

    // 根据名称查找求助帖
    @GetMapping("/findByName")
    @ResponseBody
    public List<request_resources> findByName(@RequestParam String name) {
        return requestResourcesService.findByName(name);
    }

    // 根据用户查找求助帖
    @GetMapping("/findByUserid")
    @ResponseBody
    public List<request_resources> findByUserid(@RequestParam Long userid) {
        return requestResourcesService.findByUserid(userid);
    }

    // 计算某个名称对应的总金额
    @GetMapping("/totalMoneyByName")
    @ResponseBody
    public Integer getTotalMoneyByName(@RequestParam String name) {
        return requestResourcesService.calculateTotalMoneyByName(name);
    }

    // 根据被协助用户（loaduser）查找求助帖
    @GetMapping("/findByLoaduser")
    @ResponseBody
    public List<request_resources> findByLoaduser(@RequestParam Long loaduser) {
        return requestResourcesService.findByLoaduser(loaduser);
    }

    // ✅ 新增：根据名称批量更新 loaduser 字段
    @PostMapping("/updateLoaduserByName")
    @ResponseBody
    public boolean updateLoaduserByName(@RequestParam String name, @RequestParam Long loaduser) {
        return requestResourcesService.updateLoaduserByName(name, loaduser);
    }

    // ✅ 1. 更新指定 requestid 的 torrentid
    @PutMapping("/updateTorrentid/{requestid}")
    @ResponseBody
    public boolean updateTorrentid(@PathVariable Integer requestid, @RequestParam Long torrentid) {
        return requestResourcesService.updateTorrentid(requestid, torrentid);
    }

    // ✅ 2. 查询指定 requestid 的 torrentid
    @GetMapping("/getTorrentid/{requestid}")
    @ResponseBody
    public Long getTorrentid(@PathVariable Integer requestid) {
        request_resources request = requestResourcesService.findByRequestid(requestid);
        return request != null ? request.getTorrentid() : null;
    }

    @GetMapping("/info/{requestid}")
    @ResponseBody
    public Map<String, Object> getInfoByRequestId(@PathVariable Integer requestid) {
        request_resources request = requestResourcesService.findByRequestid(requestid);
        if (request == null) {
            return null; // 或返回一个空 Map，或自定义错误处理
        }

        Map<String, Object> result = new HashMap<>();
        result.put("torrentid", request.getTorrentid());
        result.put("money", request.getMoney());
        result.put("loaduser", request.getLoaduser());
        return result;
    }


    @GetMapping("/all")
    @ResponseBody
    public List<request_resources> getAllRequests() {
        return requestResourcesService.getAllRequests();
    }

}
